<?php
/**
 * Class WPRPT_Options
 *
 * Creates a settings page for the options available for this plugin.
 *
 * @uses https://github.com/WebDevStudios/Custom-Metaboxes-and-Fields-for-WordPress
 * @see https://github.com/WebDevStudios/Custom-Metaboxes-and-Fields-for-WordPress/wiki/Using-CMB-to-create-an-Admin-Theme-Options-Page
 */
class WPRPT_Options {

	/**
	 * Generic prefix for fields and stuff
	 * @var string
	 */
	private $prefix = 'wprpt';


	/**
	 * Option key, and option page slug
	 * @var string
	 */
	private $key = 'wprpt_options';


	/**
	 * Array of metaboxes/fields
	 * @var array
	 */
	protected $option_metabox = array();


	/**
	 * Options Page title
	 * @var string
	 */
	protected $title = '';


	/**
	 * Options Page hook
	 * @var string
	 */
	protected $options_page = '';

	
	/**
	 * Constructor
	 * @since 1.0.0
	 */
	public function __construct() {
		
		// Set our title
		$this->title = __( 'Random Post Thumbnails', $this->prefix );

		// Set our CMB fields
		$this->fields = array(
			array(
				'name' => __( 'Images', $this->prefix ),
				'desc' => __( 'a variety of images to be shown at random for each post.', $this->prefix ),
				'id'   => 'images',
				'type' => 'file_list',
			),
			array(
				'name' => 'Post Types',
				'desc' => 'select the post types you\'d like to generate random thumbnails for.',
				'id' => 'selected_post_types',
				'type' => 'multicheck',
				'options' => get_post_types('', 'names')
			),
		);

		$this->fields = apply_filters( $this->key, $this->fields );
		
	}

	
	/**
	 * Initiate our hooks
	 *
	 * @since 1.0.0
	 *
	 * @param none
	 */
	public function hooks() {
		
		add_action( 'admin_init', array( $this, 'init' ) );
		add_action( 'admin_menu', array( $this, 'add_options_page' ) );

	}


	/**
	 * Register our setting to WP
	 *
	 * @since 1.0.0
	 *
	 * @param none
	 */
	public function init() {
		register_setting( $this->key, $this->key );
	}


	/**
	 * Add menu options page
	 *
	 * @since 1.0.0
	 *
	 * @param none
	 */
	public function add_options_page() {
		$this->options_page = add_options_page( $this->title, $this->title, 'manage_options', $this->key, array( $this, 'admin_page_display' ) );
	}


	/**
	 * Admin page markup. Mostly handled by CMB
	 *
	 * @since 1.0.0
	 *
	 * @param none
	 */
	public function admin_page_display() {
		?>
		<div class="wrap cmb_options_page <?php echo $this->key; ?>">
			<h2><?php echo esc_html( get_admin_page_title() ); ?></h2>
			<?php
			if ( $_POST )
				echo '<div class="message updated"><p>Settings saved!</p></div>';
			?>
			<?php cmb_metabox_form( $this->option_metabox(), $this->key ); ?>
		</div>
	<?php
	}


	/**
	 * Defines the theme option metabox and field configuration
	 *
	 * @since 1.0.0
	 *
	 * @param none
	 * @return array
	 */
	public function option_metabox() {

		return array(
			'id'         => 'option_metabox',
			'show_on'    => array( 'key' => 'options-page', 'value' => array( $this->key, ), ),
			'show_names' => true,
			'fields'     => $this->fields,
		);

	}


	/**
	 * Public getter method for retrieving protected/private variables
	 *
	 * @since 1.0.0
	 * 
	 * @param  string  $field Field to retrieve
	 * @return mixed          Field value or exception is thrown
	 */
	public function __get( $field ) {

		// Allowed fields to retrieve
		if ( in_array( $field, array( 'key', 'fields', 'title', 'options_page' ), true ) ) {
			return $this->{$field};
		}
		if ( 'option_metabox' === $field ) {
			return $this->option_metabox();
		}
		throw new Exception( 'Invalid property: ' . $field );

	}

}